package ddf.minim.ugens;

import java.util.ArrayList;
import ddf.minim.AudioOutput;
import ddf.minim.Minim;

/**
 * The UGen class is an abstract class which is intended to be the basis for
 * all UGens in Minim.  UGen is short for Unit Generator, which is simply something 
 * either generates a sample value, or transforms the sample value produced by 
 * another UGen. Because everything is a UGen, there is a common interface for patching 
 * things together. For instance, you might have a line of code that looks like this:
 * <pre>
 * osc.patch( filter ).patch( adsr ).patch( output );
 * </pre>
 * You can read this code left to right. It says that the output of an Oscil should be 
 * sent through a filter (perhaps a LowPass) and the output of the filter should be 
 * sent through an ADSR envelope, which should then be sent to an AudioOutput. It's 
 * incredibly clear what you signal path is and you can state it concisely.
 * <p>
 * UGens might also have UGenInputs. Oscil, for example, has a UGenInput called 
 * <code>frequency</code>. UGenInputs can be patched to, just like UGens, which means you
 * might have a line of code like this:
 * 
 * line.patch( osc.frequency )
 * 
 * This says that a Line UGen should control the value of the Oscil's frequency. You may have 
 * created a Line that changes it's value from 440 to 880 over 2 seconds. The audible result, 
 * when you activate() the Line, is that the Oscil will sweep upwards in frequency and then hold there 
 * until you activate the Line again. All of this control happens on a sample-by-sample basis, 
 * which means (hopefully) no clicks and pops.
 * 
 * @author Damien Di Fede, Anderson Mills
 */
public abstract class UGen
{
	/**
	 * This enum is used to specify the InputType of the UGenInput
	 * @author Anderson Mills
	 */
	// jam3: enum is automatically static so it can't be in the nested class
	public enum InputType {CONTROL, AUDIO};
	
	/**
	 * This inner class, UGenInput, is used to connect the output of other UGens to this UGen
	 * @author Anderson Mills
	 */
	// TODO make this package-protected if IIRFilter can be brougt into the ugens package
	public class UGenInput
	{
			private UGen incoming;
			private InputType inputType;
			
			/**
			 * This constructor generates an AUDIO input.
			 */
			public UGenInput()
	    {
	    	this(InputType.AUDIO);
	    }
		
	    /**
	     *  This constructor generates a UGenInput of the specified type.
	     * @param it
	     */
	    public UGenInput(InputType it)
	    {
	    	inputType = it;
    		uGenInputs.add(this);
	    }
	    
	    /**
	     *  returns the inputType
	     * @return
	     */
	    public InputType getInputType()
	    {
	    	return inputType;
	    }
	    
	    /**
	     *  returns the outer UGen of which this is an input.
	     * @return
	     */
	    public UGen getOuterUGen()
	    {
	    	return UGen.this;
	    }
	    
	    /**
	     *  returns the UGen which is giving values to this input.
	     * @return
	     */
	    public UGen getIncomingUGen()
	    {
	    	return incoming;
	    }
	    
	    /**
	     *  set the UGen which is giving values to this input
	     * @param in
	     */
	    public void setIncomingUGen(UGen in)
	    {
	    	incoming = in;
	    }
	    
	    /**
	     *  returns true if this input has an incoming UGen
	     * @return
	     */
	    public boolean isPatched()
	    {
	    	return (incoming != null);
	    }
	    
	    /**
	     *  returns the last values provided to this input from it's incoming UGen
	     * @return
	     */
			public float[] getLastValues()
			{
				return getIncomingUGen().getLastValues();
			}
			
	    /**
	     *  return the inputType as a string (for debugging)
	     * @return
	     */
	    public String getInputTypeAsString()
	    {
	    	String typeLabel = null;
	    	switch (inputType)
	    	{
	    	case AUDIO :
	    		typeLabel = "AUDIO";
	    		break;
	    	case CONTROL :
	    		typeLabel = "CONTROL";
	    		break;	
	    	}
	    	return typeLabel;
	    }
	    
	    /**
	     *  print information about this UGenInput (for debugging)
	     */
	    public void printInput()
	    {
	    	Minim.debug("UGenInput: " 
	    			+ " signal = " + getInputTypeAsString() + " " 
	    			+ isPatched() );
	    }
	}  // ends the UGenInput inner class
	
	// list of UGenInputs connected to this UGen
  private ArrayList<UGenInput> uGenInputs;
  // last values generated by this UGen
	private float[] lastValues;
	// sampleRate of this UGen
	private float sampleRate;
	// number of outputs connected to this UGen
	private int nOutputs;
	// counter for the currentTick with respect to the number of Outputs
	private int currentTick;
	
	/**
	 * Constructor for a UGen.  
	 */
	public UGen()
	{
		uGenInputs = new ArrayList<UGenInput>();
		// TODO How to set length of last values appropriately?
		// jam3: Using "2" here is wrong.  Could make ArrayList and set size with tick?
		lastValues = new float[2];
		nOutputs = 0;
		currentTick = 0;
	}
	
	/**
	 * Connect the output of this UGen to the first input of connectToUGen. Doing so will chain these 
	 * two UGens together, causing them to generate sound at the same time when the end of 
	 * chain is patched to an AudioOutput.
	 * 
	 * @param connectToUGen The UGen to connect to.
	 * @return connectToUGen is returned so that you can chain patch calls. For example: 
	 * <pre>
	 * sine.patch(gain).patch(out);
	 * </pre>
	 */
	// ddf: this is final because we never want people to override it.
	public final UGen patch(UGen connectToUGen)
	{
		// jam3: connecting to a UGen is the same as connecting to it's first input
		connectToUGen.addInput(this);
		// TODO jam3: nOutputs should only increase when this chain will be ticked!
		nOutputs += 1;
		Minim.debug("nOutputs = " + nOutputs);
   	setSampleRate( connectToUGen.sampleRate );
		return connectToUGen;
	}
	
	/**
	 * Connect the output of this UGen to a specific input of connecToUGen.
	 * @param connectToInput
	 * @return cennectToInput.getOuterUGen()
	 */
	public final UGen patch(UGenInput connectToInput)
	{
		connectToInput.setIncomingUGen(this);
		// TODO jam3: nOutputs should only increase when this chain will be ticked!
		nOutputs +=1;
		Minim.debug("nOutputs = " + nOutputs);
		setSampleRate( connectToInput.getOuterUGen().sampleRate );
		return connectToInput.getOuterUGen();
	}
	
	/**
	 * If you want to do something other than the default behavior when your UGen is patched to,
	 * you can override this method in your derived class. Summer, for instance, keeps a list 
	 * of all the UGens that have been patched to it, so that it can tick them and sum the 
	 * results when it uGenerates.
	 * 
	 * @param input
	 */
	// ddf: Protected because users of UGens should never call this directly.
	//      Sub-classes can override this to control what happens when something
	//      is patched to them. See the Summer class.
	protected void addInput(UGen input)
	{
		// jam3: This default behavior is that the incoming signal will be added
		// 		to the first input in the uGenInputs list.
		Minim.debug("UGen addInput called.");
		// TODO change input checking to an Exception?
		if (uGenInputs.size() > 0)
		{
			Minim.debug("Initializing default input on something");	
			this.uGenInputs.get(0).setIncomingUGen(input);
		}  else
		{
			System.err.println("Trying to connect to UGen with no default input.");
		}
	}
	
	/**
	 * Patch the output of this UGen to the provided AudioOuput. Doing so will immediately 
	 * result in this UGen and all UGens patched into it to begin generating audio.
	 * 
	 * @param output The AudioOutput you want to connect this UGen to.
	 */
	public final void patch(AudioOutput output)
	{
		Minim.debug("Patching " + this + " to the output " + output + ".");
		patch(output.bus);
		setSampleRate(output.sampleRate());
	}
	/**
	 * Unpatch the output of this output from the provided AudioOutput. This causes this UGen
	 * and all UGens patched into it to stop generation audio if AudioOutput is not potched 
	 * somewhere else in this chain.
	 * 
	 * @param output The AudioOutput this UGen should be disconnected from.
	 */
	public final void unpatch(AudioOutput output)
	{
		Minim.debug("Unpatching " + this + " from the output " + output + ".");
		unpatch(output.bus);
		//setSampleRate(output.sampleRate());
	}
	/**
	 * Remove this UGen as the input to the connectToUGen.
	 * @param connectToUGen
	 */
	public final void unpatch(UGen connectToUGen)
	{
		connectToUGen.removeInput(this);
		// TODO nOutputs needs to be updated as the convers of patch above.
		nOutputs -= 1;
		Minim.debug("nOutputs = " + nOutputs);
	}
	
	/**
	 * If you need to do something specific when something is unpatched from your UGen, you can 
	 * override this method. 
	 * @param input
	 */
	// This currently does nothing, but is overridden in Summer.
	protected void removeInput(UGen input)
	{
		Minim.debug("UGen removeInput called.");
		// TODO Need to set default behavior for normal UGens on removeInput
	}
	
	/**
	 * Generates one sample frame for this UGen. 
	 * 
	 * @param channels
	 *    An array that represents one sample frame. To generate a mono signal, 
	 *    pass an array of length 1, if stereo an array of length 2, and so on.
	 *    How a UGen deals with multi-channel sound will be implementation dependent.
	 */
	public final void tick(float[] channels)
	{
		if (nOutputs > 0)
		{
			// only tick once per sampleframe when multiple outputs
			currentTick = (currentTick + 1)%(nOutputs);
		}
		if (0 == currentTick) 
		{			
			if (uGenInputs.size() > 0)
			{
				for(int i=0; i<uGenInputs.size(); i++)
				{		
					if ((uGenInputs.get(i) != null) && (uGenInputs.get(i).isPatched()))
					{
						float[] tmp;
						switch (uGenInputs.get(i).inputType)
						{
						case CONTROL :
							tmp = new float[1];
							break;
						default : // includes AUDIO
							tmp = new float[channels.length];
							break;
						}
						// tick all connected inputs
						uGenInputs.get(i).getIncomingUGen().tick(tmp);
					}
				}
			}
			// and then uGenerate for this UGen	
			uGenerate(channels);
			// need to keep the last values generated so we have something to hand multiple outputs 
			System.arraycopy(channels, 0, lastValues, 0, channels.length);
		}
	}
	
	/**
	 * Implement this method when you extend UGen. It will be called when your UGen 
	 * needs to generate one sample frame of audio. It is expected that you will 
	 * assign values to the array and <em>not</em> simply modify the existing 
	 * values. In the case where you write a UGen that takes audio input and 
	 * modifies it, the pattern to follow is to have the first UGenInput you create 
	 * be your audio input and then in uGenerate you will use the <code>getLastValues</code>
	 * method of your audio UGenInput to retrieve the audio you want to modify, 
	 * which you will then modify however 
	 * you need to, assigning the result to the values in <code>channels</code>.
	 * @param channels an array representing one sample frame. 
	 */
	protected abstract void uGenerate(float[] channels);
	
	/**
	 * Return the last values generated by this UGen. This will most often be used 
	 * by sub-classes when pulling data from their inputs.
	 * 
	 */
	public final float[] getLastValues()
	{
		return lastValues;
	}
	
	/**
	 * Returns the sample rate of this UGen.
	 */
	public final float sampleRate()
	{
		return sampleRate;
	}
	
	/**
	 * Override this method in your derived class to receive a notification
	 * when the sample rate of your UGen has changed. You might need to do 
	 * this to recalculate sample rate dependent values, such as the 
	 * step size for an oscillator.
	 *
	 */
	protected void sampleRateChanged()
	{
		// default implementation does nothing.
	}
	
	/**
	 * Set the sample rate for this UGen.
	 * 
	 * @param newSampleRate the sample rate this UGen should generate at.
	 */
	// ddf: changed this to public because Summer needs to be able to call it
	//      on all of its UGens when it has its sample rate set by being connected 
	//      to an AudioOuput. Realized it's not actually a big deal for people to 
	//      set the sample rate of any UGen they create whenever they want. In fact, 
	//      could actually make total sense to want to do this with something playing 
	//      back a chunk of audio loaded from disk. Made this final because it should 
	//      never be overridden. If sub-classes need to know about sample rate changes 
	//      the should override sampleRateChanged()
	public final void setSampleRate(float newSampleRate)
	{
		if (sampleRate != newSampleRate)
		{
			sampleRate = newSampleRate;
			sampleRateChanged();
		}
		if (uGenInputs.size() > 0)
		{
			for(int i=0; i<uGenInputs.size(); i++)
			{		
				if ((uGenInputs.get(i) != null) && (uGenInputs.get(i).isPatched()))
				{
					uGenInputs.get(i).getIncomingUGen().setSampleRate(newSampleRate);
				}
			}			
		}
	}	
	
	/**
	 *  Prints all inputs connected to this UGen (for debugging)
	 */
	public void printInputs()
	{
	   for(int i=0; i<uGenInputs.size(); i++)
	   {
		   Minim.debug("uGenInputs " + i + " ");
		   if (uGenInputs.get(i) == null)
		   {
			   Minim.debug("null");   
		   } else {
			   uGenInputs.get(i).printInput();
		   }
	   }
	}
}
